// WebSocket Client Singleton
let ws = null;
let listeners = {}; // Event listeners

export function connect() {
    const protocol = window.location.protocol === 'https:' ? 'wss:' : 'ws:';
    const wsUrl = `${protocol}//${window.location.host}/ws`;
    
    ws = new WebSocket(wsUrl);

    ws.onopen = () => {
        emit('open');
    };

    ws.onmessage = (event) => {
        try {
            const msg = JSON.parse(event.data);
            emit('message', msg);
        } catch (e) {
            console.error('Parse error:', e);
        }
    };

    ws.onclose = () => {
        emit('close');
        setTimeout(connect, 3000); // Auto reconnect
    };

    ws.onerror = (err) => {
        console.error('Socket error:', err);
        ws.close();
    };
}

export function send(data) {
    if (ws && ws.readyState === WebSocket.OPEN) {
        if (data instanceof Blob) {
            ws.send(data);
        } else {
            ws.send(JSON.stringify(data));
        }
        return true;
    }
    return false;
}

export function getReadyState() {
    return ws ? ws.readyState : WebSocket.CLOSED;
}

export function getBufferedAmount() {
    return ws ? ws.bufferedAmount : 0;
}

// Simple Event Bus
export function on(event, callback) {
    if (!listeners[event]) listeners[event] = [];
    listeners[event].push(callback);
}

function emit(event, data) {
    if (listeners[event]) {
        listeners[event].forEach(cb => cb(data));
    }
}
