import { send } from '../core/websocket.js';
import { els } from './ui.js';

let enabled = false;
let penColor = '#ff0000';
let penWidth = 3;
let lastPoint = null;
let moveThrottle = null;

function resizeCanvas() {
    if (!els.previewContainer) return;
    const rect = els.previewContainer.getBoundingClientRect();
    const canvas = document.getElementById('draw-canvas');
    if (!canvas) return;
    canvas.width = Math.floor(rect.width);
    canvas.height = Math.floor(rect.height);
    canvas.style.width = rect.width + 'px';
    canvas.style.height = rect.height + 'px';
}

function getCtx() {
    const canvas = document.getElementById('draw-canvas');
    if (!canvas) return null;
    return canvas.getContext('2d');
}

function drawLocalLine(p1, p2) {
    const ctx = getCtx();
    if (!ctx) return;
    
    if (penColor === 'eraser') {
        ctx.globalCompositeOperation = 'destination-out';
        ctx.lineWidth = 20; // Eraser size
    } else {
        ctx.globalCompositeOperation = 'source-over';
        ctx.strokeStyle = penColor;
        ctx.lineWidth = penWidth;
    }
    
    ctx.lineCap = 'round';
    ctx.lineJoin = 'round';
    ctx.beginPath();
    ctx.moveTo(p1.x, p1.y);
    ctx.lineTo(p2.x, p2.y);
    ctx.stroke();
}

function sendPoint(state, x, y) {
    send({
        action: 'draw',
        data: {
            type: 'path',
            state,
            x,
            y,
            color: penColor,
            width: penColor === 'eraser' ? 20 : penWidth
        }
    });
}

function handleTouchStart(e) {
    if (!enabled) return;
    const rect = els.previewContainer.getBoundingClientRect();
    
    // Get image rect for accurate coordinate mapping
    const imgRect = els.imgEl.getBoundingClientRect();
    // Use image rect if available and visible, otherwise fallback to container
    const targetRect = (imgRect.width > 0 && imgRect.height > 0) ? imgRect : rect;

    if (e.touches.length >= 1) {
        const t = e.touches[0];
        
        // Coordinates for local drawing (relative to container/canvas)
        const x = t.clientX - rect.left;
        const y = t.clientY - rect.top;
        
        lastPoint = { x, y };
        
        // Coordinates for remote sending (relative to image)
        const relX = t.clientX - targetRect.left;
        const relY = t.clientY - targetRect.top;
        
        const xr = relX / targetRect.width;
        const yr = relY / targetRect.height;
        
        sendPoint('start', xr, yr);
        e.preventDefault();
    }
}

function handleTouchMove(e) {
    if (!enabled || !lastPoint) return;
    const rect = els.previewContainer.getBoundingClientRect();
    
    // Get image rect for accurate coordinate mapping
    const imgRect = els.imgEl.getBoundingClientRect();
    const targetRect = (imgRect.width > 0 && imgRect.height > 0) ? imgRect : rect;

    if (e.touches.length >= 1) {
        const t = e.touches[0];
        
        // Local drawing
        const x = t.clientX - rect.left;
        const y = t.clientY - rect.top;
        const nowPoint = { x, y };
        drawLocalLine(lastPoint, nowPoint);
        lastPoint = nowPoint;
        
        if (!moveThrottle) {
            moveThrottle = setTimeout(() => {
                // Remote sending
                const relX = t.clientX - targetRect.left;
                const relY = t.clientY - targetRect.top;
                
                const xr = relX / targetRect.width;
                const yr = relY / targetRect.height;
                
                sendPoint('move', xr, yr);
                moveThrottle = null;
            }, 10);
        }
        e.preventDefault();
    }
}

function handleTouchEnd(e) {
    if (!enabled) return;
    const rect = els.previewContainer.getBoundingClientRect();
    const imgRect = els.imgEl.getBoundingClientRect();
    const targetRect = (imgRect.width > 0 && imgRect.height > 0) ? imgRect : rect;

    if (lastPoint) {
        // We use the last known point's relative position? 
        // Actually handleTouchEnd doesn't provide new coordinates usually.
        // But we need to send 'end' signal.
        // We can just send 0,0 or the last normalized coords. 
        // The backend 'end' event mainly stops the line. 
        // But let's try to be consistent if possible, or just send what we have.
        // Re-calculating from lastPoint (which is local) to relative is hard without keeping track of targetRect.
        // But sendPoint 'end' ignores x/y in many implementations, or just uses them as final point.
        // Let's recalculate from lastPoint if possible, but lastPoint is relative to rect, not targetRect.
        
        // Easier: Just send 'end' with 0,0 or last valid.
        // PC side implementation of 'end':
        // elif state == 'end': self.last_x = None; self.last_y = None
        // So x/y doesn't matter for 'end'.
        
        sendPoint('end', 0, 0);
        lastPoint = null;
    }
}

export function initDrawing() {
    const canvas = document.getElementById('draw-canvas');
    if (!canvas) return;
    resizeCanvas();
    window.addEventListener('resize', resizeCanvas);
    canvas.addEventListener('touchstart', handleTouchStart, { passive: false });
    canvas.addEventListener('touchmove', handleTouchMove, { passive: false });
    canvas.addEventListener('touchend', handleTouchEnd);
    canvas.addEventListener('touchcancel', handleTouchEnd);
}

export function toggleDrawing() {
    enabled = !enabled;
    const canvas = document.getElementById('draw-canvas');
    if (!canvas) return;
    
    // UI Updates
    if (els.btnDraw) {
        if (enabled) {
            els.btnDraw.classList.add('btn-active');
        } else {
            els.btnDraw.classList.remove('btn-active');
        }
    }

    if (enabled) {
        resizeCanvas();
    }
    
    if (els.lockInd) {
        els.lockInd.style.display = enabled ? 'flex' : 'none';
    }
    
    if (els.colorPicker) {
        els.colorPicker.style.display = enabled ? 'flex' : 'none';
    }

    canvas.style.pointerEvents = enabled ? 'auto' : 'none';
    const ctx = getCtx();
    if (ctx) {
        ctx.clearRect(0, 0, canvas.width, canvas.height);
    }
    
    // Send clear command to remote
    if (!enabled) {
        send({
            action: 'draw',
            data: {
                type: 'clear'
            }
        });
    }
}

export function setPen(color, width) {
    if (color) penColor = color;
    if (width) penWidth = width;
}

export function isDrawingEnabled() {
    return enabled;
}

export function disableDrawing() {
    enabled = false;
    const btnDraw = document.getElementById('btn-draw');
    const colorPicker = document.getElementById('color-picker');
    const lockInd = document.getElementById('lock-indicator');
    
    if (btnDraw) btnDraw.classList.remove('btn-active');
    if (colorPicker) colorPicker.style.display = 'none';
    if (lockInd) lockInd.style.display = 'none';
}

export function saveDrawing() {
    const img = document.getElementById('preview-img');
    const canvas = document.getElementById('draw-canvas');
    if (!img || !canvas) return;

    // Create a temp canvas with the same size as the original image
    const tempCanvas = document.createElement('canvas');
    // If naturalWidth is 0 (e.g. image not loaded), fallback to displayed width
    tempCanvas.width = img.naturalWidth || img.width || 800;
    tempCanvas.height = img.naturalHeight || img.height || 600;
    const ctx = tempCanvas.getContext('2d');

    // Draw the image
    // Note: img.src might be a blob or data URL.
    ctx.drawImage(img, 0, 0, tempCanvas.width, tempCanvas.height);

    // Draw the drawing canvas on top, scaling it to fit
    // canvas.width/height corresponds to screen pixels (set in resizeCanvas)
    // We scale it to match tempCanvas size
    ctx.drawImage(canvas, 0, 0, tempCanvas.width, tempCanvas.height);

    // Trigger download
    try {
        const dataURL = tempCanvas.toDataURL('image/png');
        const link = document.createElement('a');
        const now = new Date();
        const timestamp = now.getFullYear() +
            String(now.getMonth() + 1).padStart(2, '0') +
            String(now.getDate()).padStart(2, '0') + '_' +
            String(now.getHours()).padStart(2, '0') +
            String(now.getMinutes()).padStart(2, '0') +
            String(now.getSeconds()).padStart(2, '0');
        
        link.download = 'ppt-note-' + timestamp + '.png';
        link.href = dataURL;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    } catch (e) {
        console.error('Save failed:', e);
        alert('保存失败，请重试');
    }
}
