import { send } from '../core/websocket.js';
import { els } from './ui.js';

// State
let currentImgAngle = 0;
let currentImgScale = 1.0;
let currentTranslateX = 0;
let currentTranslateY = 0;

// Gesture State
let isGesturing = false;
let startTouches = [];
let startScale = 1.0;
let startAngle = 0;
let startTranslateX = 0;
let startTranslateY = 0;

// Sync State
let lastSentScale = 1.0;
let lastSentAngle = 0;
let syncTimer = null;

export function initImageControl() {
    // Touch Events
    if (els.gestureArea) {
        els.gestureArea.addEventListener('touchstart', handleTouchStart, {passive: false});
        els.gestureArea.addEventListener('touchmove', handleTouchMove, {passive: false});
        els.gestureArea.addEventListener('touchend', handleTouchEnd);
        els.gestureArea.addEventListener('touchcancel', handleTouchEnd);
    }
    
    // File Upload
    if (els.fileUpload) {
        els.fileUpload.addEventListener('change', handleFileUpload);
    }
}

export function showImageView(src) {
    els.viewImg.src = src;
    els.imageViewOverlay.style.display = 'flex';
    
    // Reset State
    currentImgAngle = 0;
    currentImgScale = 1.0;
    currentTranslateX = 0;
    currentTranslateY = 0;
    lastSentScale = 1.0;
    lastSentAngle = 0;
    
    updateImageView(false);
}

export function closeImageView() {
    els.imageViewOverlay.style.display = 'none';
    els.viewImg.src = '';
    send({action: "image_control", cmd: "close"});
}

export function controlImage(cmd, value) {
    if (cmd === 'rotate') {
        currentImgAngle = (currentImgAngle + value) % 360;
        lastSentAngle = currentImgAngle;
        updateImageView(true);
    } else if (cmd === 'zoom') {
        currentImgScale *= value;
        lastSentScale = currentImgScale;
        updateImageView(true);
    }

    send({
        action: "image_control", 
        cmd: cmd, 
        value: value
    });
}

function updateImageView(animate = false) {
    els.viewImg.style.transition = animate ? 'transform 0.2s' : 'none';
    els.viewImg.style.transform = `translate(${currentTranslateX}px, ${currentTranslateY}px) rotate(${currentImgAngle}deg) scale(${currentImgScale})`;
}

// ... Touch handling functions ...
function handleTouchStart(e) {
    if (e.touches.length >= 1) {
        e.preventDefault();
        isGesturing = true;
        startTouches = Array.from(e.touches);
        startScale = currentImgScale;
        startAngle = currentImgAngle;
        startTranslateX = currentTranslateX;
        startTranslateY = currentTranslateY;
    }
}

function handleTouchMove(e) {
    if (!isGesturing) return;
    e.preventDefault();
    const touches = Array.from(e.touches);
    
    // Pan
    if (touches.length === 1 && startTouches.length === 1) {
        const dx = touches[0].clientX - startTouches[0].clientX;
        const dy = touches[0].clientY - startTouches[0].clientY;
        currentTranslateX = startTranslateX + dx;
        currentTranslateY = startTranslateY + dy;
        updateImageView(false);
        return;
    }
    
    // Pinch/Rotate
    if (touches.length >= 2 && startTouches.length >= 2) {
        const dist = getDistance(touches[0], touches[1]);
        const angle = getAngle(touches[0], touches[1]);
        const startDist = getDistance(startTouches[0], startTouches[1]);
        const startAng = getAngle(startTouches[0], startTouches[1]);
        
        if (startDist > 0) {
            currentImgScale = startScale * (dist / startDist);
        }
        currentImgAngle = startAngle + (angle - startAng);
        
        const center = getCenter(touches[0], touches[1]);
        const startCenter = getCenter(startTouches[0], startTouches[1]);
        currentTranslateX = startTranslateX + (center.x - startCenter.x);
        currentTranslateY = startTranslateY + (center.y - startCenter.y);

        updateImageView(false);
        throttleSync();
    }
}

function handleTouchEnd(e) {
    if (e.touches.length === 0) {
        isGesturing = false;
        startTouches = [];
        syncToPC();
    } else {
        startTouches = Array.from(e.touches);
        startScale = currentImgScale;
        startAngle = currentImgAngle;
        startTranslateX = currentTranslateX;
        startTranslateY = currentTranslateY;
    }
}

function getDistance(t1, t2) {
    const dx = t1.clientX - t2.clientX;
    const dy = t1.clientY - t2.clientY;
    return Math.sqrt(dx * dx + dy * dy);
}

function getAngle(t1, t2) {
    const dx = t1.clientX - t2.clientX;
    const dy = t1.clientY - t2.clientY;
    return Math.atan2(dy, dx) * 180 / Math.PI;
}

function getCenter(t1, t2) {
    return { x: (t1.clientX + t2.clientX) / 2, y: (t1.clientY + t2.clientY) / 2 };
}

function throttleSync() {
    if (!syncTimer) {
        syncTimer = setTimeout(() => {
            syncToPC();
            syncTimer = null;
        }, 50);
    }
}

function syncToPC() {
    // Logic similar to original syncToPC
    let scaleDelta = 1.0;
    if (lastSentScale > 0) scaleDelta = currentImgScale / lastSentScale;
    const angleDelta = currentImgAngle - lastSentAngle;
    
    if (Math.abs(scaleDelta - 1.0) > 0.01) {
        send({action: "image_control", cmd: "zoom", value: scaleDelta});
        lastSentScale = currentImgScale;
    }
    if (Math.abs(angleDelta) > 0.5) {
        send({action: "image_control", cmd: "rotate", value: angleDelta});
        lastSentAngle = currentImgAngle;
    }
}

export function triggerUpload() {
    els.fileUpload.click();
}

async function handleFileUpload(e) {
    const file = e.target.files[0];
    if (!file) return;

    const objectUrl = URL.createObjectURL(file);
    showImageView(objectUrl);

    // Update text span only, preserving the icon
    const spanEl = els.btnUpload.querySelector('span');
    const originalText = spanEl ? spanEl.innerText : '拍照';
    
    if (spanEl) spanEl.innerText = '⏳ 上传中...';
    els.btnUpload.style.pointerEvents = 'none'; // Disable clicks

    const formData = new FormData();
    formData.append('file', file);

    try {
        const response = await fetch('/api/upload', {
            method: 'POST',
            body: formData
        });

        if (response.ok) {
            if (spanEl) spanEl.innerText = '上传成功';
            setTimeout(() => {
                if (spanEl) spanEl.innerText = originalText;
                els.btnUpload.style.pointerEvents = 'auto';
            }, 2000);
        } else {
            alert('上传失败');
            if (spanEl) spanEl.innerText = originalText;
            els.btnUpload.style.pointerEvents = 'auto';
        }
    } catch (error) {
        console.error(error);
        alert('上传出错');
        if (spanEl) spanEl.innerText = originalText;
        els.btnUpload.style.pointerEvents = 'auto';
    } finally {
        e.target.value = '';
    }
}
