import { send, getBufferedAmount } from '../core/websocket.js';
import { els } from './ui.js';

let isLive = false;
let liveStream = null;
let liveInterval = null;
const ctx = els.canvasEl ? els.canvasEl.getContext('2d') : null;

export async function toggleLive() {
    if (!isLive) {
        // Start Live
        try {
            const constraints = { 
                video: { 
                    facingMode: 'environment',
                    width: { ideal: 1280 },
                    height: { ideal: 720 }
                }, 
                audio: false 
            };
            
            liveStream = await navigator.mediaDevices.getUserMedia(constraints);
            els.videoEl.srcObject = liveStream;
            
            els.videoEl.onloadedmetadata = () => {
                isLive = true;
                els.liveOverlay.style.display = 'flex';
                
                send({action: "start_live"});
                startSendingFrames();
            };
        } catch (err) {
            console.error("Camera Error:", err);
            alert("无法获取摄像头权限或设备不支持: " + err.message);
        }
    } else {
        stopLive();
    }
}

export function stopLive(isRemote = false) {
    if (!isRemote && isLive) {
        send({action: "stop_live"});
    }

    isLive = false;
    els.liveOverlay.style.display = 'none';
    
    if (liveInterval) {
        clearInterval(liveInterval);
        liveInterval = null;
    }
    
    if (liveStream) {
        liveStream.getTracks().forEach(track => track.stop());
        liveStream = null;
    }
    
    els.videoEl.srcObject = null;
}

function startSendingFrames() {
    liveInterval = setInterval(() => {
        if (!isLive) return;
        
        // Backpressure check (64KB)
        if (getBufferedAmount() > 64 * 1024) return;
        
        const MAX_LONG_SIDE = 1280;
        const vw = els.videoEl.videoWidth;
        const vh = els.videoEl.videoHeight;
        
        if (vw && vh) {
            let w = vw;
            let h = vh;
            
            if (w > h) {
                if (w > MAX_LONG_SIDE) {
                    h = Math.round(h * (MAX_LONG_SIDE / w));
                    w = MAX_LONG_SIDE;
                }
            } else {
                if (h > MAX_LONG_SIDE) {
                    w = Math.round(w * (MAX_LONG_SIDE / h));
                    h = MAX_LONG_SIDE;
                }
            }
            
            if (els.canvasEl.width !== w || els.canvasEl.height !== h) {
                els.canvasEl.width = w;
                els.canvasEl.height = h;
            }
            
            ctx.drawImage(els.videoEl, 0, 0, w, h);
            
            els.canvasEl.toBlob((blob) => {
                if (blob) send(blob);
            }, 'image/jpeg', 0.7);
        }
    }, 50);
}
