import { send } from '../core/websocket.js';
import { els, showFeedback, showLoading, hideLoading } from './ui.js';
import { isDrawingEnabled, disableDrawing } from './drawing.js';

let currentTotal = 0;
let initialLoadDone = false;

export function initPPT() {
    // Click Navigation
    if (els.previewContainer) {
        els.previewContainer.addEventListener('click', function(e) {
            if (els.playStatusEl.classList.contains('disconnected')) return;

            const rect = els.previewContainer.getBoundingClientRect();
            const x = e.clientX - rect.left;
            const width = rect.width;
            
            if (x < width / 2) {
                sendCommand('prev');
                showFeedback(els.feedbackLeft);
            } else {
                sendCommand('next');
                showFeedback(els.feedbackRight);
            }
        });
    }

    // Double tap prevention
    document.addEventListener('dblclick', function(event) {
        event.preventDefault();
    }, { passive: false });
}

export function sendCommand(action, params = {}) {
    if (navigator.vibrate) navigator.vibrate(10);
    
    // Lock navigation when drawing is enabled
    if (isDrawingEnabled() && ['prev', 'next', 'goto'].includes(action)) {
        return;
    }
    
    if (action === 'start_show') {
        showLoading('正在启动放映...');
        // Safety timeout
        setTimeout(() => {
            if (els.loadingOverlay.style.display !== 'none' && els.loadingOverlayText.innerText.includes('启动')) {
                hideLoading();
            }
        }, 8000);
    }

    send({ action: action, ...params });
}

export function handleUpdate(msg) {
    // Hide initial loading
    if (!initialLoadDone) {
        initialLoadDone = true;
        setTimeout(hideLoading, 1500);
    }

    // Hide start show loading
    if (els.loadingOverlay.style.display !== 'none' && els.loadingOverlayText.innerText.includes('启动')) {
        if (msg.status === 'playing' || msg.status === 'stopped') {
            hideLoading();
        }
    }

    els.loadingEl.style.display = 'none';
    
    if (msg.status === 'no_ppt' || msg.status === 'no_office_running' || msg.status === 'connection_failed' || !msg.image) {
        els.imgEl.style.display = 'none';
        els.loadingEl.style.display = 'block';
        
        if (msg.status === 'no_office_running') {
            els.loadingEl.innerText = '未检测到 Office 软件运行\n请打开 WPS 或 PowerPoint';
        } else if (msg.status === 'connection_failed') {
            els.loadingEl.innerHTML = '检测到软件运行，但连接失败<br/><br/><button class="btn btn-system" style="background:#f44336; margin:0 auto; display:inline-block;" onclick="window.app.sendCommand(\'repair\')">🛠️ 一键修复 (强行关闭软件)</button>';
        } else {
            els.loadingEl.innerText = '未检测到打开的 PPT\n请在电脑上打开文件';
        }
        
        els.pageInfoEl.innerText = '-- / --';
        els.playStatusEl.innerText = '未就绪';
        els.playStatusEl.classList.remove('playing');
        els.stripEl.innerHTML = '';
        return;
    }

    els.imgEl.style.display = 'block';
    els.imgEl.src = msg.image;
    els.pageInfoEl.innerText = `${msg.current_index} / ${msg.total_count}`;
    
    updateStrip(msg.total_count, msg.current_index);

    // Update Status
    if (msg.status === 'playing') {
        els.playStatusEl.innerText = '放映中';
        els.playStatusEl.classList.add('playing');
        
        // Remove inactive filter
        els.imgEl.classList.remove('inactive');
        els.stripEl.classList.remove('inactive');
        
        if (els.btnDraw) els.btnDraw.style.display = 'flex';
    } else {
        els.playStatusEl.innerText = '未放映';
        els.playStatusEl.classList.remove('playing');
        
        // Add inactive filter
        els.imgEl.classList.add('inactive');
        els.stripEl.classList.add('inactive');
        
        if (els.btnDraw) els.btnDraw.style.display = 'none';
        disableDrawing();
    }

    // Update App Type
    if (msg.app_type) {
        els.appTypeEl.style.display = 'inline-block';
        els.appTypeEl.innerText = msg.app_type;
        if (msg.app_type === 'PowerPoint') {
            els.appTypeEl.style.backgroundColor = '#d24726';
        } else {
            els.appTypeEl.style.backgroundColor = '#0078d4';
        }
    }

    if (els.lockInd) {
        if (msg.status === 'playing' && isDrawingEnabled()) {
            els.lockInd.style.display = 'flex';
        } else {
            els.lockInd.style.display = 'none';
        }
    }
}

function updateStrip(total, current) {
    if (total !== currentTotal) {
        currentTotal = total;
        els.stripEl.innerHTML = '';
        for (let i = 1; i <= total; i++) {
            const img = document.createElement('img');
            img.className = 'slide-thumb';
            img.loading = "lazy";
            img.src = `/api/slides/${i}`; 
            img.onclick = () => sendCommand('goto', { index: i });
            img.id = `thumb-${i}`;
            els.stripEl.appendChild(img);
        }
    }
    
    document.querySelectorAll('.slide-thumb').forEach(el => el.classList.remove('active'));
    const activeThumb = document.getElementById(`thumb-${current}`);
    if (activeThumb) {
        activeThumb.classList.add('active');
        activeThumb.scrollIntoView({ behavior: 'smooth', block: 'nearest', inline: 'center' });
    }
}
